/*
 * MIT License - AVL Heap Test Program (Updated for configurable balance bits)
 * Combines demonstration and API testing with support for TOP and BOTTOM balance bits
 *
 * Compile: gcc -Wall -o test_avl avl_heap.c test_avl_heap.c -lm
 */

#include "avl_heap.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>

/* ANSI color codes for visualization */
const char* level_colors[] = {
    "\033[38;5;0m",    /* Level 0: Black */
    "\033[38;5;94m",   /* Level 1: Brown */
    "\033[38;5;196m",  /* Level 2: Red */
    "\033[38;5;208m",  /* Level 3: Orange */
};
const char* color_reset = "\033[0m";

/* Forward declarations for helper functions */
void run_basic_api_tests(void);
void run_balance_bit_config_tests(void);
void run_visualization_demo(void);
int get_level(size_t idx);
void visualize_heap_compact(AVLHeap *avlh);

/* Calculate tree level from array index */
int get_level(size_t idx)
{
    if (idx == 0) return 0;

    int level = 0;
    size_t val = idx + 1;
    while (val > 1) {
        val >>= 1;
        level++;
    }
    return level;
}

void visualize_heap_compact(AVLHeap *avlh)
{
    printf("\n=== Heap Visualization ===\n");
    printf("Note: Full visualization requires internal heap access.\n");
    printf("Using library traversal instead:\n\n");
    avlh_print_inorder(avlh);
}

/*---------------------------------------------------------------------------
 * Test 1: Key-Only Heap with TOP bits
 *---------------------------------------------------------------------------*/
void test_key_only_heap_top_bits(void)
{
    printf("╔════════════════════════════════════════════════════════════════╗\n");
    printf("║ TEST 1: Key-Only Heap (TOP bits - for integers)               ║\n");
    printf("╚════════════════════════════════════════════════════════════════╝\n");

    AVLHeap *ids = avlh_init("UserIDs", 16, 1, 0, 0, BALANCE_TOP_BITS);

    printf("Adding IDs: 100, 200, 150, 175, 125\n");
    avlh_add(ids, 100, NULL);
    avlh_add(ids, 200, NULL);
    avlh_add(ids, 150, NULL);
    avlh_add(ids, 175, NULL);
    avlh_add(ids, 125, NULL);

    printf("\nFind tests:\n");
    printf("  ID 150 exists? %s\n", avlh_find(ids, 150) ? "YES" : "NO");
    printf("  ID 999 exists? %s\n", avlh_find(ids, 999) ? "YES" : "NO");

    printf("\nAll IDs in order:\n");
    avlh_print_inorder(ids);

    printf("\nDeleting ID 150...\n");
    if (avlh_delete(ids, 150)) {
        printf("  Success! ID 150 removed\n");
    }

    printf("\nAfter deletion:\n");
    avlh_print_inorder(ids);

    avlh_stats(ids);
    avlh_free(ids);
    printf("\n");
}

/*---------------------------------------------------------------------------
 * Test 2: Key-Value Heap with TOP bits
 *---------------------------------------------------------------------------*/
void test_key_value_heap_top_bits(void)
{
    printf("╔════════════════════════════════════════════════════════════════╗\n");
    printf("║ TEST 2: Key-Value Heap (TOP bits - for integers)              ║\n");
    printf("╚════════════════════════════════════════════════════════════════╝\n");

    AVLHeap *salaries = avlh_init("Salaries", 16, 2, 0, 0, BALANCE_TOP_BITS);

    printf("Adding employee salaries (EmployeeID -> Salary):\n");
    uint64_t data1[] = {75000};
    uint64_t data2[] = {120000};
    uint64_t data3[] = {95000};

    avlh_add(salaries, 1001, data1);  /* Employee 1001: $75,000 */
    avlh_add(salaries, 1005, data2);  /* Employee 1005: $120,000 */
    avlh_add(salaries, 1003, data3);  /* Employee 1003: $95,000 */

    printf("\nFind employee 1003's salary:\n");
    uint64_t *salary = avlh_find_element(salaries, 1003);
    if (salary) {
        printf("  Employee 1003 earns: $%llu\n", (unsigned long long)salary[0]);
    }

    printf("\nAll salaries in order:\n");
    avlh_print_inorder(salaries);

    avlh_stats(salaries);
    avlh_free(salaries);
    printf("\n");
}

/*---------------------------------------------------------------------------
 * Test 3: Pointer-like values with BOTTOM bits
 *---------------------------------------------------------------------------*/
void test_pointer_heap_bottom_bits(void)
{
    printf("╔════════════════════════════════════════════════════════════════╗\n");
    printf("║ TEST 3: Pointer-like Heap (BOTTOM bits - for aligned addrs)   ║\n");
    printf("╚════════════════════════════════════════════════════════════════╝\n");

    /* Use bottom 2 bits for balance, simulate pointer addresses (4-byte aligned) */
    AVLHeap *ptrs = avlh_init("Pointers", 16, 1, 0, 0, BALANCE_BOTTOM_BITS);

    printf("Adding simulated pointer addresses (4-byte aligned):\n");
    /* All values must be 4-byte aligned (bottom 2 bits = 0) */
    avlh_add(ptrs, 0x1000, NULL);
    avlh_add(ptrs, 0x2000, NULL);
    avlh_add(ptrs, 0x1800, NULL);
    avlh_add(ptrs, 0x2400, NULL);
    avlh_add(ptrs, 0x1400, NULL);

    printf("\nFind tests:\n");
    printf("  Address 0x1800 exists? %s\n", avlh_find(ptrs, 0x1800) ? "YES" : "NO");
    printf("  Address 0x9999 exists? %s\n", avlh_find(ptrs, 0x9999) ? "YES" : "NO");

    printf("\nTrying to add non-aligned address (should fail):\n");
    int result = avlh_add(ptrs, 0x1001, NULL);  /* Not 4-byte aligned */
    printf("  Add 0x1001: %s (expected: FAILED)\n", result ? "SUCCESS" : "FAILED");

    printf("\nAll addresses in order:\n");
    avlh_print_inorder(ptrs);

    avlh_stats(ptrs);
    avlh_free(ptrs);
    printf("\n");
}

/*---------------------------------------------------------------------------
 * Test 4: Multi-word with balance in different positions
 *---------------------------------------------------------------------------*/
void test_multiword_balance_positions(void)
{
    printf("╔════════════════════════════════════════════════════════════════╗\n");
    printf("║ TEST 4: Multi-word elements with different balance positions  ║\n");
    printf("╚════════════════════════════════════════════════════════════════╝\n");

    /* Test 1: Balance in first word (word 0), TOP bits */
    printf("\n--- Configuration 1: Balance in word 0, TOP bits ---\n");
    AVLHeap *heap1 = avlh_init("Config1", 16, 3, 0, 0, BALANCE_TOP_BITS);
    uint64_t data1[] = {1111, 2222};
    uint64_t data2[] = {3333, 4444};
    avlh_add(heap1, 100, data1);
    avlh_add(heap1, 200, data2);
    avlh_print_inorder(heap1);
    avlh_free(heap1);

    /* Test 2: Balance in last word (word 2), TOP bits */
    printf("\n--- Configuration 2: Balance in word 2 (last), TOP bits ---\n");
    AVLHeap *heap2 = avlh_init("Config2", 16, 3, 0, 2, BALANCE_TOP_BITS);
    uint64_t data3[] = {5555, 6666};
    uint64_t data4[] = {7777, 8888};
    avlh_add(heap2, 300, data3);
    avlh_add(heap2, 400, data4);
    avlh_print_inorder(heap2);
    avlh_free(heap2);

    /* Test 3: Balance in middle word (word 1), BOTTOM bits */
    printf("\n--- Configuration 3: Balance in word 1 (middle), BOTTOM bits ---\n");
    AVLHeap *heap3 = avlh_init("Config3", 16, 3, 0, 1, BALANCE_BOTTOM_BITS);
    uint64_t data5[] = {9999, 1234};
    uint64_t data6[] = {5678, 4321};
    avlh_add(heap3, 0x1000, data5);  /* Must be aligned for BOTTOM bits */
    avlh_add(heap3, 0x2000, data6);
    avlh_print_inorder(heap3);
    avlh_free(heap3);

    printf("\n");
}

/*---------------------------------------------------------------------------
 * Test 5: Large key values
 *---------------------------------------------------------------------------*/
void test_large_keys(void)
{
    printf("╔════════════════════════════════════════════════════════════════╗\n");
    printf("║ TEST 5: Large key values                                      ║\n");
    printf("╚════════════════════════════════════════════════════════════════╝\n");

    /* TOP bits: can use keys up to 2^62 - 1 */
    printf("\n--- TOP bits: Maximum usable range ---\n");
    AVLHeap *heap_top = avlh_init("TopBits", 16, 1, 0, 0, BALANCE_TOP_BITS);
    
    uint64_t max_top_key = (1ULL << 62) - 1;
    printf("  Adding max valid key: %llu (2^62 - 1)\n", (unsigned long long)max_top_key);
    int result1 = avlh_add(heap_top, max_top_key, NULL);
    printf("  Result: %s\n", result1 ? "SUCCESS" : "FAILED");
    
    printf("  Trying key >= 2^62 (should fail):\n");
    int result2 = avlh_add(heap_top, 1ULL << 62, NULL);
    printf("  Result: %s (expected: FAILED)\n", result2 ? "SUCCESS" : "FAILED");
    
    avlh_free(heap_top);

    /* BOTTOM bits: can use any 4-byte aligned value, but uses more of the address space */
    printf("\n--- BOTTOM bits: Can use larger absolute values (if aligned) ---\n");
    AVLHeap *heap_bottom = avlh_init("BottomBits", 16, 1, 0, 0, BALANCE_BOTTOM_BITS);
    
    uint64_t large_aligned = 0xFFFFFFFFFFFFFFFCULL;  /* Max value with bottom 2 bits = 0 */
    printf("  Adding large aligned value: 0x%llX\n", (unsigned long long)large_aligned);
    int result3 = avlh_add(heap_bottom, large_aligned, NULL);
    printf("  Result: %s\n", result3 ? "SUCCESS" : "FAILED");
    
    avlh_free(heap_bottom);

    printf("\n");
}

/*---------------------------------------------------------------------------
 * Main Test Runner
 *---------------------------------------------------------------------------*/
void run_basic_api_tests(void)
{
    printf("\n");
    printf("═══════════════════════════════════════════════════════════════════\n");
    printf("   AVL HEAP LIBRARY - COMPREHENSIVE API TESTS                      \n");
    printf("   Testing configurable balance bit positions                      \n");
    printf("═══════════════════════════════════════════════════════════════════\n");
    printf("\n");

    test_key_only_heap_top_bits();
    test_key_value_heap_top_bits();
    test_pointer_heap_bottom_bits();
    test_multiword_balance_positions();
    test_large_keys();

    printf("╔════════════════════════════════════════════════════════════════╗\n");
    printf("║ ALL TESTS COMPLETED                                            ║\n");
    printf("╚════════════════════════════════════════════════════════════════╝\n");
    printf("\n");
}

int main(void)
{
    run_basic_api_tests();
    return 0;
}
