/*
 * MIT License
 *
 * Copyright (c) 2026
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#ifndef AVL_HEAP_H
#define AVL_HEAP_H

#include <stdint.h>
#include <stddef.h>

#define GiB (1ULL << 30)

/* Balance bit location specification */
typedef enum {
    BALANCE_TOP_BITS,    /* Use bits 62-63 (MSB) - for integers */
    BALANCE_BOTTOM_BITS  /* Use bits 0-1 (LSB) - for pointer-like values */
} BalanceBitLocation;

/* Opaque structure - implementation details hidden */
typedef struct AVLHeap AVLHeap;

/*
 * Initialize a new AVL heap
 * 
 * Parameters:
 *   name              - Descriptive name for debugging
 *   initial_capacity  - Initial number of elements to allocate
 *   element_words     - Number of 64-bit words per element (minimum 1)
 *   max_memory        - Maximum arena size (0 = default 1 GiB)
 *   balance_word_idx  - Index of word (0 to element_words-1) containing balance bits
 *   balance_location  - BALANCE_TOP_BITS or BALANCE_BOTTOM_BITS
 * 
 * Returns: Heap pointer or NULL on failure
 */
AVLHeap* avlh_init(const char *name, size_t initial_capacity, 
                   size_t element_words, size_t max_memory,
                   size_t balance_word_idx, BalanceBitLocation balance_location);

/*
 * Free all memory associated with a heap
 */
void avlh_free(AVLHeap *avlh);

/*
 * Add element to heap
 * 
 * Parameters:
 *   avlh - Heap to add to
 *   key  - Key value (constraints depend on balance bit location:
 *          TOP_BITS: must be < 2^62
 *          BOTTOM_BITS: must be 4-byte aligned, i.e., bottom 2 bits must be 0)
 *   data - Pointer to data array (NULL for key-only heaps)
 * 
 * Returns: 1 on success, 0 on failure
 */
int avlh_add(AVLHeap *avlh, uint64_t key, const uint64_t *data);

/*
 * Check if key exists
 * 
 * Returns: 1 if found, 0 if not found
 */
int avlh_find(AVLHeap *avlh, uint64_t key);

/*
 * Find element and return pointer to its data
 * 
 * Returns: Pointer to data array or NULL
 */
uint64_t* avlh_find_element(AVLHeap *avlh, uint64_t key);

/*
 * Delete element by key
 * 
 * Returns: 1 on success, 0 if not found
 */
int avlh_delete(AVLHeap *avlh, uint64_t key);

/*
 * Print elements in sorted order
 */
void avlh_print_inorder(AVLHeap *avlh);

/*
 * Print elements in pre-order (root first)
 */
void avlh_print_preorder(AVLHeap *avlh);

/*
 * Print elements in post-order (leaves first)
 */
void avlh_print_postorder(AVLHeap *avlh);

/*
 * Print detailed heap statistics
 */
void avlh_stats(AVLHeap *avlh);

/*
 * Get current size of heap
 */
size_t avlh_get_size(AVLHeap *avlh);

#endif /* AVL_HEAP_H */
