# AVL Heap with Configurable Balance Bits

## Overview

This implementation of an AVL heap allows you to configure which bits of which 64-bit word are used to store the AVL tree balance information. This flexibility enables the heap to work efficiently with different data types:

- **Integer keys**: Use the top 2 bits (bits 62-63) which aren't typically used
- **Pointer-like values**: Use the bottom 2 bits (bits 0-1) which are often free due to alignment

## API Changes

### New Enum: `BalanceBitLocation`

```c
typedef enum {
    BALANCE_TOP_BITS,    /* Use bits 62-63 (MSB) - for integers */
    BALANCE_BOTTOM_BITS  /* Use bits 0-1 (LSB) - for pointer-like values */
} BalanceBitLocation;
```

### Updated `avlh_init` Function

```c
AVLHeap* avlh_init(const char *name, 
                   size_t initial_capacity, 
                   size_t element_words, 
                   size_t max_memory,
                   size_t balance_word_idx,        /* NEW */
                   BalanceBitLocation balance_location);  /* NEW */
```

**New Parameters:**
- `balance_word_idx`: Index of the word (0 to element_words-1) that contains the balance bits
- `balance_location`: Either `BALANCE_TOP_BITS` or `BALANCE_BOTTOM_BITS`

## Usage Examples

### Example 1: Integer Keys (TOP bits)

For standard integer keys, use the top 2 bits:

```c
AVLHeap *ids = avlh_init("UserIDs", 16, 1, 0, 
                         0,                    // balance bits in word 0
                         BALANCE_TOP_BITS);    // use bits 62-63

avlh_add(ids, 100, NULL);  // Keys must be < 2^62
avlh_add(ids, 200, NULL);
```

**Constraint**: Keys must be less than 2^62 (4,611,686,018,427,387,904)

### Example 2: Pointer-like Values (BOTTOM bits)

For 4-byte aligned pointers or addresses, use the bottom 2 bits:

```c
AVLHeap *ptrs = avlh_init("Pointers", 16, 1, 0,
                          0,                     // balance bits in word 0
                          BALANCE_BOTTOM_BITS);  // use bits 0-1

avlh_add(ptrs, 0x1000, NULL);  // Must be 4-byte aligned
avlh_add(ptrs, 0x2000, NULL);  // Bottom 2 bits must be 0
```

**Constraint**: Keys must be 4-byte aligned (bottom 2 bits = 0)

### Example 3: Multi-word Elements

For multi-word elements, you can place balance bits in any word:

```c
// Balance in first word
AVLHeap *h1 = avlh_init("Config1", 16, 3, 0, 
                        0,                 // balance in word 0
                        BALANCE_TOP_BITS);

// Balance in last word
AVLHeap *h2 = avlh_init("Config2", 16, 3, 0,
                        2,                 // balance in word 2 (last)
                        BALANCE_TOP_BITS);

// Balance in middle word with bottom bits
AVLHeap *h3 = avlh_init("Config3", 16, 3, 0,
                        1,                   // balance in word 1 (middle)
                        BALANCE_BOTTOM_BITS);
```

## Use Cases

### 1. Standard Integer Keys (TOP bits)
Best for:
- Sequential IDs
- Counters
- Any integer values < 2^62

### 2. Memory Addresses (BOTTOM bits)
Best for:
- Pointers (naturally 4-byte or 8-byte aligned)
- Memory addresses
- File offsets (often aligned)
- Any values where low bits are unused due to alignment

### 3. Mixed Data Structures
With multi-word elements, you can:
- Store the key in one word (with balance bits)
- Store associated data in other words
- Choose balance bit location based on key characteristics

## Key Constraints Summary

| Configuration | Constraint | Usable Range |
|--------------|-----------|--------------|
| BALANCE_TOP_BITS | key < 2^62 | 0 to 4,611,686,018,427,387,903 |
| BALANCE_BOTTOM_BITS | key % 4 == 0 | Any 4-byte aligned value |

## Implementation Details

### Internal Structure

The `AVLHeap` structure now includes:

```c
struct AVLHeap {
    size_t balance_word_idx;           // Which word has balance bits
    BalanceBitLocation balance_location; // TOP or BOTTOM bits
    uint64_t balance_mask;             // Mask for balance bits
    uint64_t value_mask;               // Mask for key value
    int balance_shift;                 // Shift amount
    // ... other fields
};
```

### Bit Masks

```c
// For BALANCE_TOP_BITS:
balance_mask = 0xC000000000000000  // bits 62-63
value_mask   = 0x3FFFFFFFFFFFFFFF  // bits 0-61
balance_shift = 62

// For BALANCE_BOTTOM_BITS:
balance_mask = 0x0000000000000003  // bits 0-1
value_mask   = 0xFFFFFFFFFFFFFFFC  // bits 2-63
balance_shift = 0
```

## Testing

The `test_avl_heap.c` file includes comprehensive tests:

1. **Test 1**: Key-only heap with TOP bits (integers)
2. **Test 2**: Key-value heap with TOP bits 
3. **Test 3**: Pointer-like heap with BOTTOM bits
4. **Test 4**: Multi-word elements with different balance positions
5. **Test 5**: Large key values and boundary conditions

Compile and run:
```bash
gcc -Wall -o test_avl avl_heap.c test_avl_heap.c -lm
./test_avl
```

## Migration from Previous Version

Old code:
```c
AVLHeap *heap = avlh_init("MyHeap", 16, 1, 0);
```

New code (equivalent behavior):
```c
AVLHeap *heap = avlh_init("MyHeap", 16, 1, 0,
                          0,                // balance in word 0
                          BALANCE_TOP_BITS); // use top bits
```

## Performance Considerations

- **Memory**: No additional memory overhead
- **Speed**: Negligible performance difference between TOP and BOTTOM bit configurations
- **Flexibility**: Choose the configuration that best matches your data alignment

## Error Handling

The `avlh_add` function validates keys based on the configuration:

```c
int result = avlh_add(heap, key, data);
// Returns 1 on success, 0 on failure
```

Failures occur when:
- TOP bits: key >= 2^62
- BOTTOM bits: key & 0x3 != 0 (not 4-byte aligned)
