# AVL Heap Library - Quick Compilation Guide

## Files Included

1. **avl_heap.h** - Library header file (public API)
2. **avl_heap.c** - Library implementation
3. **test_avl_heap.c** - Comprehensive test program
4. **Makefile** - Build automation
5. **BUILD.md** - Detailed build instructions

## Quick Start - Simple Compilation

### Single Command (Fastest)

```bash
gcc -Wall -o test_avl avl_heap.c test_avl_heap.c -lm
./test_avl
```

This compiles both the library and test program together and runs the tests.

## Using the Makefile (Recommended)

```bash
# Build everything
make

# Run tests
make test

# Clean up
make clean
```

## Build Static Library

If you want to build a reusable static library:

```bash
# Method 1: Using Make
make lib

# Method 2: Manual
gcc -Wall -c -o avl_heap.o avl_heap.c
ar rcs libavlheap.a avl_heap.o

# Then compile your program against it
gcc -Wall -o my_program my_program.c -L. -lavlheap -lm
```

## Summary of Changes from Original

### What Was Separated

The original `avl_heap_demo.c` (867 lines) was split into:

1. **Library Code (avl_heap.c + avl_heap.h)**
   - All AVL heap implementation
   - Memory arena management
   - Public API functions (add, find, delete, traversals, stats)
   - ~470 lines of reusable library code

2. **Test Code (test_avl_heap.c)**
   - Combines original demo functionality
   - Adds test suite from test_avl_heap.c
   - Demonstrates all library features
   - ~520 lines of test/demo code

### Key Benefits

✅ **Reusable Library**: Use the AVL heap in any project
✅ **Clean API**: Simple, well-documented interface
✅ **Comprehensive Tests**: All features tested
✅ **Easy Compilation**: Multiple build methods
✅ **Well Documented**: API reference, build guide, examples

## Example Usage in Your Code

```c
#include "avl_heap.h"

int main() {
    // Create heap: name, capacity, words_per_element, max_memory
    AVLHeap *heap = avlh_init("MyData", 16, 2, 0);
    
    // Add key-value pairs
    uint64_t value = 12345;
    avlh_add(heap, 100, &value);
    
    // Find data
    if (avlh_find(heap, 100)) {
        uint64_t *data = avlh_find_element(heap, 100);
        printf("Found: %llu\n", *data);
    }
    
    // Delete
    avlh_delete(heap, 100);
    
    // Print all in order
    avlh_print_inorder(heap);
    
    // Clean up
    avlh_free(heap);
    return 0;
}
```

Compile with:
```bash
gcc -o my_program my_program.c avl_heap.c -lm
```

## Requirements

- GCC 4.8+ or compatible C compiler
- POSIX-compliant OS (Linux, macOS, BSD)
- Math library (libm) - usually included

## Test Output

The test program runs two comprehensive suites:

1. **API Test Suite**: Basic operations on small datasets
2. **Demo Suite**: 100-element heaps with different configurations

Both demonstrate:
- Key-only heaps (8 bytes/element)
- Key-value heaps (16 bytes/element)
- Multi-value heaps (40 bytes/element)
- Find, delete, and traversal operations
- Memory statistics

## Troubleshooting

### Error: "undefined reference to 'log2'"
**Solution**: Add `-lm` flag
```bash
gcc -o test_avl avl_heap.c test_avl_heap.c -lm
```

### Error: "mmap: Cannot allocate memory"
**Solution**: Reduce max_memory parameter
```c
AVLHeap *heap = avlh_init("MyHeap", 16, 1, 100*1024*1024); // 100 MB
```

### Warnings about printf format specifiers
These are harmless - uint64_t format varies by platform. The code works correctly.

## Next Steps

1. See **BUILD.md** for detailed compilation options
2. See **API_REFERENCE.md** for complete API documentation
3. Run the tests to see all features in action
4. Use the library in your own projects!

---

For questions or issues, refer to the comprehensive documentation in BUILD.md and API_REFERENCE.md.
