# AVL Heap Library - Build Instructions

This package contains the AVL heap data structure separated into a reusable library.

## File Structure

```
avl_heap.h                  - Public API header
avl_heap.c                  - Library implementation
test_avl_heap.c             - Test program
Makefile                    - Build automation
BUILD.md                    - This file
```

## Quick Start

### Option 1: Using Makefile (Recommended)

```bash
# Build everything
make

# Run tests
make test

# Clean up
make clean
```

### Option 2: Manual Compilation

#### Build and Run Tests (Simple Method)

```bash
# Compile library and test together
gcc -Wall -o test_avl avl_heap.c test_avl_heap.c -lm

# Run tests
./test_avl
```

#### Build as Static Library (Advanced)

```bash
# Step 1: Compile library to object file
gcc -Wall -c -o avl_heap.o avl_heap.c

# Step 2: Create static library archive
ar rcs libavlheap.a avl_heap.o

# Step 3: Compile test program linking against library
gcc -Wall -o test_avl test_avl_heap.c -L. -lavlheap -lm

# Step 4: Run tests
./test_avl
```

## Compilation Options

### Debug Build

```bash
gcc -Wall -g -DDEBUG -o test_avl avl_heap.c test_avl_heap.c -lm
```

### Optimized Release Build

```bash
gcc -Wall -O3 -DNDEBUG -o test_avl avl_heap.c test_avl_heap.c -lm
```

### With All Warnings

```bash
gcc -Wall -Wextra -Wpedantic -o test_avl avl_heap.c test_avl_heap.c -lm
```

## Using the Library in Your Own Programs

### Method 1: Link Against Static Library

```bash
# First build the library
gcc -Wall -c -o avl_heap.o avl_heap.c
ar rcs libavlheap.a avl_heap.o

# Then compile your program
gcc -Wall -o my_program my_program.c -L. -lavlheap -lm
```

### Method 2: Include Source Directly

```bash
gcc -Wall -o my_program my_program.c avl_heap.c -lm
```

### Example Program

```c
#include "avl_heap.h"
#include <stdio.h>

int main() {
    // Create heap
    AVLHeap *heap = avlh_init("MyHeap", 16, 2, 0);
    
    // Add data
    uint64_t value = 42;
    avlh_add(heap, 100, &value);
    
    // Find data
    uint64_t *found = avlh_find_element(heap, 100);
    if (found) {
        printf("Found: %llu\n", *found);
    }
    
    // Clean up
    avlh_free(heap);
    return 0;
}
```

Compile:
```bash
gcc -Wall -o my_program my_program.c avl_heap.c -lm
```

## Dependencies

- **Compiler**: GCC 4.8+ or Clang 3.4+
- **Libraries**: 
  - libc (standard C library)
  - libm (math library, for log2 calculations)
- **OS**: Linux, macOS, or any POSIX-compliant system with mmap support

## Platform Notes

### Linux
Standard compilation works as-is.

### macOS
Standard compilation works as-is. May need Xcode Command Line Tools:
```bash
xcode-select --install
```

### Windows (MinGW/Cygwin)
Should work with minor modifications. May need to adjust mmap calls.

### Windows (MSVC)
Requires porting mmap to Windows VirtualAlloc API.

## Troubleshooting

### "undefined reference to `log2`"
Add `-lm` to link the math library:
```bash
gcc -o test_avl avl_heap.c test_avl_heap.c -lm
```

### "mmap: Cannot allocate memory"
Reduce the max_memory parameter when calling `avlh_init()`:
```c
AVLHeap *heap = avlh_init("MyHeap", 16, 1, 100*1024*1024); // 100 MB instead of 1 GB
```

### Library not found when using -lavlheap
Make sure the library path is included:
```bash
gcc -o test_avl test_avl_heap.c -L. -lavlheap -lm
```

Or set LD_LIBRARY_PATH:
```bash
export LD_LIBRARY_PATH=.:$LD_LIBRARY_PATH
```

## Test Output

The test program runs two suites:

1. **API Test Suite**: Tests basic operations
   - Key-only heaps
   - Key-value heaps  
   - Multi-value heaps
   - Bulk operations with 1000 elements

2. **Demonstration Suite**: Shows variable element sizes
   - Three heaps with different element sizes
   - Find and delete operations
   - Tree traversals (in-order, pre-order, post-order)
   - Memory efficiency comparison

Expected output includes:
- Colored terminal output (works in xterm-256 compatible terminals)
- Statistics for each heap
- Verification of all operations
- Memory usage information

## Installation

To install the library system-wide (requires root):

```bash
# Install header
sudo cp avl_heap.h /usr/local/include/

# Install library
sudo cp libavlheap.a /usr/local/lib/

# Update library cache (Linux)
sudo ldconfig
```

Then use in your programs:
```c
#include <avl_heap.h>
```

Compile:
```bash
gcc -o my_program my_program.c -lavlheap -lm
```

## API Documentation

See `API_REFERENCE.md` for complete API documentation including:
- Function signatures
- Parameter descriptions
- Return values
- Usage examples
- Performance characteristics

## License

MIT License - See source files for full license text.
